"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.useGlobalLoadingState = exports.useGlobalSubmittingState = exports.useGlobalPendingState = exports.useGlobalTransitionStates = void 0;
const react_1 = require("@remix-run/react");
const react_2 = require("react");
/**
 * This is a helper hook that returns the state of every fetcher active on
 * the app and combine it with the state of the global transition.
 * @example
 * let states = useGlobalTransitionStates();
 * if (state.includes("loading")) {
 *   // The app is loading.
 * }
 * if (state.includes("submitting")) {
 *   // The app is submitting.
 * }
 * // The app is idle
 */
function useGlobalTransitionStates() {
    let transition = (0, react_1.useTransition)();
    let fetchers = (0, react_1.useFetchers)();
    /**
     * This gets the state of every fetcher active on the app and combine it with
     * the state of the global transition (Link and Form).
     */
    return (0, react_2.useMemo)(function getGlobalTransitionStates() {
        return [transition.state, ...fetchers.map((fetcher) => fetcher.state)];
    }, [transition.state, fetchers]);
}
exports.useGlobalTransitionStates = useGlobalTransitionStates;
/**
 * Let you know if the app is pending some request, either global transition
 * or some fetcher transition.
 * @returns "idle" | "pending"
 */
function useGlobalPendingState() {
    let isSubmitting = useGlobalSubmittingState() === "submitting";
    let isLoading = useGlobalLoadingState() === "loading";
    if (isLoading || isSubmitting)
        return "pending";
    return "idle";
}
exports.useGlobalPendingState = useGlobalPendingState;
/**
 * Let you know if the app is submitting some request, either global transition
 * or some fetcher transition.
 * @returns "idle" | "submitting"
 */
function useGlobalSubmittingState() {
    let states = useGlobalTransitionStates();
    if (states.includes("submitting"))
        return "submitting";
    return "idle";
}
exports.useGlobalSubmittingState = useGlobalSubmittingState;
/**
 * Let you know if the app is loading some request, either global transition
 * or some fetcher transition.
 * @returns "idle" | "loading"
 */
function useGlobalLoadingState() {
    let states = useGlobalTransitionStates();
    if (states.includes("loading"))
        return "loading";
    return "idle";
}
exports.useGlobalLoadingState = useGlobalLoadingState;
