import { CookieParseOptions, CookieSerializeOptions, SessionStorage } from "@remix-run/server-runtime";
import { z } from "zod";
export interface TypedSession<Schema extends z.ZodTypeAny> {
    /**
     * Marks a session as a typed session.
     */
    readonly isTyped: boolean;
    /**
     * A unique identifier for this session.
     *
     * Note: This will be the empty string for newly created sessions and
     * sessions that are not backed by a database (i.e. cookie-based sessions).
     */
    readonly id: string;
    /**
     * The raw data contained in this session.
     *
     * This is useful mostly for SessionStorage internally to access the raw
     * session data to persist.
     */
    readonly data: z.infer<Schema>;
    /**
     * Returns `true` if the session has a value for the given `name`, `false`
     * otherwise.
     */
    has<Key extends keyof z.infer<Schema>>(name: Key): boolean;
    /**
     * Returns the value for the given `name` in this session.
     */
    get<Key extends keyof z.infer<Schema>>(key: Key): z.infer<Schema>[Key] | null;
    /**
     * Sets a value in the session for the given `name`.
     */
    set<Key extends keyof z.infer<Schema>>(name: Key, value: z.infer<Schema>[Key]): void;
    /**
     * Sets a value in the session that is only valid until the next `get()`.
     * This can be useful for temporary values, like error messages.
     */
    flash<Key extends keyof z.infer<Schema>>(name: Key, value: z.infer<Schema>[Key]): void;
    /**
     * Removes a value from the session.
     */
    unset<Key extends keyof z.infer<Schema>>(name: Key): void;
}
export interface TypedSessionStorage<Schema extends z.ZodTypeAny> {
    getSession(cookieHeader?: string | null | undefined, options?: CookieParseOptions | undefined): Promise<TypedSession<Schema>>;
    commitSession(session: TypedSession<Schema>, options?: CookieSerializeOptions | undefined): Promise<string>;
    destroySession(session: TypedSession<Schema>, options?: CookieSerializeOptions | undefined): Promise<string>;
}
export declare function createTypedSessionStorage<Schema extends z.AnyZodObject>({ sessionStorage, schema, }: {
    sessionStorage: SessionStorage;
    schema: Schema;
}): TypedSessionStorage<Schema>;
/**
 * ReReturns true if an object is a Remix Utils typed session.
 *
 * @see https://github.com/sergiodxa/remix-utils#typed-session
 */
export declare function isTypedSession<Schema extends z.AnyZodObject>(value: unknown): value is TypedSession<Schema>;
