"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.image = exports.txt = exports.xml = exports.html = exports.pdf = exports.stylesheet = exports.javascript = exports.notModified = exports.serverError = exports.unprocessableEntity = exports.notFound = exports.forbidden = exports.unauthorized = exports.badRequest = exports.redirectBack = exports.created = void 0;
const server_runtime_1 = require("@remix-run/server-runtime");
/**
 * Create a response receiving a JSON object with the status code 201.
 * @example
 * export async function action({ request }: ActionArgs) {
 *   let result = await doSomething(request);
 *   return created(result);
 * }
 */
function created(data, init) {
    return (0, server_runtime_1.json)(data, { ...init, status: 201 });
}
exports.created = created;
/**
 * Create a new Response with a redirect set to the URL the user was before.
 * It uses the Referer header to detect the previous URL. It asks for a fallback
 * URL in case the Referer couldn't be found, this fallback should be a URL you
 * may be ok the user to land to after an action even if it's not the same.
 * @example
 * export async function action({ request }: ActionArgs) {
 *   await doSomething(request);
 *   // If the user was on `/search?query=something` we redirect to that URL
 *   // but if we couldn't we redirect to `/search`, which is an good enough
 *   // fallback
 *   return redirectBack(request, { fallback: "/search" });
 * }
 */
function redirectBack(request, { fallback, ...init }) {
    var _a;
    return (0, server_runtime_1.redirect)((_a = request.headers.get("Referer")) !== null && _a !== void 0 ? _a : fallback, init);
}
exports.redirectBack = redirectBack;
/**
 * Create a response receiving a JSON object with the status code 400.
 * @example
 * export async function loader({ request }: LoaderArgs) {
 *   let user = await getUser(request);
 *   throw badRequest<BoundaryData>({ user });
 * }
 */
function badRequest(data, init) {
    return (0, server_runtime_1.json)(data, { ...init, status: 400 });
}
exports.badRequest = badRequest;
/**
 * Create a response receiving a JSON object with the status code 401.
 * @example
 * export async function loader({ request }: LoaderArgs) {
 *   let user = await getUser(request);
 *   throw unauthorized<BoundaryData>({ user });
 * }
 */
function unauthorized(data, init) {
    return (0, server_runtime_1.json)(data, { ...init, status: 401 });
}
exports.unauthorized = unauthorized;
/**
 * Create a response receiving a JSON object with the status code 403.
 * @example
 * export async function loader({ request }: LoaderArgs) {
 *   let user = await getUser(request);
 *   if (!user.idAdmin) throw forbidden<BoundaryData>({ user });
 * }
 */
function forbidden(data, init) {
    return (0, server_runtime_1.json)(data, { ...init, status: 403 });
}
exports.forbidden = forbidden;
/**
 * Create a response receiving a JSON object with the status code 404.
 * @example
 * export async function loader({ request, params }: LoaderArgs) {
 *   let user = await getUser(request);
 *   if (!db.exists(params.id)) throw notFound<BoundaryData>({ user });
 * }
 */
function notFound(data, init) {
    return (0, server_runtime_1.json)(data, { ...init, status: 404 });
}
exports.notFound = notFound;
/**
 * Create a response receiving a JSON object with the status code 422.
 * @example
 * export async function loader({ request, params }: LoaderArgs) {
 *   let user = await getUser(request);
 *   throw unprocessableEntity<BoundaryData>({ user });
 * }
 */
function unprocessableEntity(data, init) {
    return (0, server_runtime_1.json)(data, { ...init, status: 422 });
}
exports.unprocessableEntity = unprocessableEntity;
/**
 * Create a response receiving a JSON object with the status code 500.
 * @example
 * export async function loader({ request }: LoaderArgs) {
 *   let user = await getUser(request);
 *   throw serverError<BoundaryData>({ user });
 * }
 */
function serverError(data, init) {
    return (0, server_runtime_1.json)(data, { ...init, status: 500 });
}
exports.serverError = serverError;
/**
 * Create a response with only the status 304 and optional headers.
 * This is useful when trying to implement conditional responses based on Etags.
 * @example
 * export async function loader({ request }: LoaderArgs) {
 *   return notModified();
 * }
 */
function notModified(init) {
    return new Response("", { ...init, status: 304 });
}
exports.notModified = notModified;
/**
 * Create a response with a JavaScript file response.
 * It receives a string with the JavaScript content and set the Content-Type
 * header to `application/javascript; charset=utf-8` always.
 *
 * This is useful to dynamically create a JS file from a Resource Route.
 * @example
 * export async function loader({ request }: LoaderArgs) {
 *   return javascript("console.log('Hello World')");
 * }
 */
function javascript(content, init = {}) {
    let responseInit = typeof init === "number" ? { status: init } : init;
    let headers = new Headers(responseInit.headers);
    if (!headers.has("Content-Type")) {
        headers.set("Content-Type", "application/javascript; charset=utf-8");
    }
    return new Response(content, {
        ...responseInit,
        headers,
    });
}
exports.javascript = javascript;
/**
 * Create a response with a CSS file response.
 * It receives a string with the CSS content and set the Content-Type header to
 * `text/css; charset=utf-8` always.
 *
 * This is useful to dynamically create a CSS file from a Resource Route.
 * @example
 * export async function loader({ request }: LoaderArgs) {
 *   return css("body { color: red; }");
 * }
 */
function stylesheet(content, init = {}) {
    let responseInit = typeof init === "number" ? { status: init } : init;
    let headers = new Headers(responseInit.headers);
    if (!headers.has("Content-Type")) {
        headers.set("Content-Type", "text/css; charset=utf-8");
    }
    return new Response(content, {
        ...responseInit,
        headers,
    });
}
exports.stylesheet = stylesheet;
/**
 * Create a response with a PDF file response.
 * It receives a string with the PDF content and set the Content-Type header to
 * `application/pdf; charset=utf-8` always.
 *
 * This is useful to dynamically create a PDF file from a Resource Route.
 * @example
 * export async function loader({ request }: LoaderArgs) {
 *   return pdf(await generatePDF(request.formData()));
 * }
 */
function pdf(content, init = {}) {
    let responseInit = typeof init === "number" ? { status: init } : init;
    let headers = new Headers(responseInit.headers);
    if (!headers.has("Content-Type")) {
        headers.set("Content-Type", "application/pdf");
    }
    return new Response(content, {
        ...responseInit,
        headers,
    });
}
exports.pdf = pdf;
/**
 * Create a response with a HTML file response.
 * It receives a string with the HTML content and set the Content-Type header to
 * `text/html; charset=utf-8` always.
 *
 * This is useful to dynamically create a HTML file from a Resource Route.
 * @example
 * export async function loader({ request }: LoaderArgs) {
 *   return html("<h1>Hello World</h1>");
 * }
 */
function html(content, init = {}) {
    let responseInit = typeof init === "number" ? { status: init } : init;
    let headers = new Headers(responseInit.headers);
    if (!headers.has("Content-Type")) {
        headers.set("Content-Type", "text/html; charset=utf-8");
    }
    return new Response(content, {
        ...responseInit,
        headers,
    });
}
exports.html = html;
/**
 * Create a response with a XML file response.
 * It receives a string with the XML content and set the Content-Type header to
 * `application/xml; charset=utf-8` always.
 *
 * This is useful to dynamically create a XML file from a Resource Route.
 * @example
 * export let loader: LoaderFunction = async ({ request }) => {
 *   return xml("<?xml version='1.0'?><catalog></catalog>");
 * }
 */
function xml(content, init = {}) {
    let responseInit = typeof init === "number" ? { status: init } : init;
    let headers = new Headers(responseInit.headers);
    if (!headers.has("Content-Type")) {
        headers.set("Content-Type", "application/xml; charset=utf-8");
    }
    return new Response(content, {
        ...responseInit,
        headers,
    });
}
exports.xml = xml;
/**
 * Create a response with a TXT file response.
 * It receives a string with the TXT content and set the Content-Type header to
 * `text/plain; charset=utf-8` always.
 *
 * This is useful to dynamically create a TXT file from a Resource Route.
 * @example
 * export let loader: LoaderFunction = async ({ request }) => {
 *   return txt(`
 *     User-agent: *
 *     Allow: /
 *   `);
 * }
 */
function txt(content, init = {}) {
    let responseInit = typeof init === "number" ? { status: init } : init;
    let headers = new Headers(responseInit.headers);
    if (!headers.has("Content-Type")) {
        headers.set("Content-Type", "text/plain; charset=utf-8");
    }
    return new Response(content, {
        ...responseInit,
        headers,
    });
}
exports.txt = txt;
/**
 * Create a response with a image file response.
 * It receives a Buffer, ArrayBuffer or ReadableStream with the image content
 * and set the Content-Type header to the `type` parameter.
 *
 * This is useful to dynamically create a image file from a Resource Route.
 * @example
 * export async function loader({ request }: LoaderArgs) {
 *   return image(await takeScreenshot(), { type: "image/avif" });
 * }
 */
function image(content, { type, ...init }) {
    let headers = new Headers(init.headers);
    if (!headers.has("Content-Type")) {
        headers.set("Content-Type", type);
    }
    return new Response(content, {
        ...init,
        headers,
    });
}
exports.image = image;
