/**
 * @remix-run/express v1.19.3
 *
 * Copyright (c) Remix Software Inc.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE.md file in the root directory of this source tree.
 *
 * @license MIT
 */
'use strict';

Object.defineProperty(exports, '__esModule', { value: true });

var node = require('@remix-run/node');

/**
 * A function that returns the value to use as `context` in route `loader` and
 * `action` functions.
 *
 * You can think of this as an escape hatch that allows you to pass
 * environment/platform-specific values through to your loader/action, such as
 * values that are generated by Express middleware like `req.session`.
 */

/**
 * Returns a request handler for Express that serves the response using Remix.
 */
function createRequestHandler({
  build,
  getLoadContext,
  mode = process.env.NODE_ENV
}) {
  let handleRequest = node.createRequestHandler(build, mode);
  return async (req, res, next) => {
    try {
      let request = createRemixRequest(req, res);
      let loadContext = await (getLoadContext === null || getLoadContext === void 0 ? void 0 : getLoadContext(req, res));
      let response = await handleRequest(request, loadContext);
      await sendRemixResponse(res, response);
    } catch (error) {
      // Express doesn't support async functions, so we have to pass along the
      // error manually using next().
      next(error);
    }
  };
}
function createRemixHeaders(requestHeaders) {
  let headers = new node.Headers();
  for (let [key, values] of Object.entries(requestHeaders)) {
    if (values) {
      if (Array.isArray(values)) {
        for (let value of values) {
          headers.append(key, value);
        }
      } else {
        headers.set(key, values);
      }
    }
  }
  return headers;
}
function createRemixRequest(req, res) {
  let url = new URL(`${req.protocol}://${req.get("host")}${req.url}`);

  // Abort action/loaders once we can no longer write a response
  let controller = new node.AbortController();
  res.on("close", () => controller.abort());
  let init = {
    method: req.method,
    headers: createRemixHeaders(req.headers),
    // Cast until reason/throwIfAborted added
    // https://github.com/mysticatea/abort-controller/issues/36
    signal: controller.signal
  };
  if (req.method !== "GET" && req.method !== "HEAD") {
    init.body = req;
  }
  return new node.Request(url.href, init);
}
async function sendRemixResponse(res, nodeResponse) {
  res.statusMessage = nodeResponse.statusText;
  res.status(nodeResponse.status);
  for (let [key, values] of Object.entries(nodeResponse.headers.raw())) {
    for (let value of values) {
      res.append(key, value);
    }
  }
  if (nodeResponse.body) {
    await node.writeReadableStreamToWritable(nodeResponse.body, res);
  } else {
    res.end();
  }
}

exports.createRemixHeaders = createRemixHeaders;
exports.createRemixRequest = createRemixRequest;
exports.createRequestHandler = createRequestHandler;
exports.sendRemixResponse = sendRemixResponse;
